# دليل تنفيذ نظام إدارة الشحنات - Implementation Guide

## البنية الأساسية

بما أن النظام كبير جداً، سيتم تقسيمه إلى مراحل. هذا الدليل يوضح كيفية البدء.

## المرحلة 1: الجداول الأساسية

ابدأ بإنشاء الجداول الأساسية التالية:

1. `courier_companies` - شركات الشحن
2. `shipment_orders` - طلبات الشحن
3. `shipment_items` - عناصر الشحنة
4. `sellers` - البائعون
5. `delivery_locations` - مواقع التسليم

## المرحلة 2: النماذج والعلاقات

بعد إنشاء الجداول، قم بإنشاء Models مع العلاقات:

```php
// ShipmentOrder Model
public function items()
{
    return $this->hasMany(ShipmentItem::class);
}

public function courierCompany()
{
    return $this->belongsTo(CourierCompany::class);
}

public function seller()
{
    return $this->belongsTo(Seller::class);
}
```

## المرحلة 3: المتحكمات (Controllers)

أنشئ Controllers للمكونات الرئيسية:

- `ShipmentOrderController`
- `CourierCompanyController`
- `SellerController`
- `InventoryController`
- `ManifestController`

## المرحلة 4: المسارات (Routes)

أضف المسارات في `routes/web.php`:

```php
Route::middleware(['auth', 'verified'])->group(function () {
    // Shipment Management Routes
    Route::resource('shipment-orders', ShipmentOrderController::class);
    Route::resource('courier-companies', CourierCompanyController::class);
    Route::resource('sellers', SellerController::class);
    // ... المزيد
});
```

## المرحلة 5: الواجهات (Views)

أنشئ Views باستخدام Blade Templates في `resources/views/shipment/`

## نصائح التنفيذ

1. **ابدأ بالمراحل الأساسية**: ركز على الطلبات (Orders) أولاً
2. **استخدم النمط الموجود**: اتبع نفس نمط WarehouseController و Warehouse Model
3. **الصلاحيات**: أضف permissions جديدة في النظام
4. **DataTables**: استخدم Yajra DataTables للقوائم
5. **التدريج**: قم بالتنفيذ على مراحل لتسهيل الاختبار

## الملفات المطلوبة

للبدء السريع، تحتاج لإنشاء:

1. Migrations (في database/migrations)
2. Models (في app/Models)
3. Controllers (في app/Http/Controllers)
4. Views (في resources/views/shipment/)
5. Routes (في routes/web.php)
6. DataTables (في app/DataTables)

## مثال على Migration

```php
<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        if (!Schema::hasTable('shipment_orders'))
        {
            Schema::create('shipment_orders', function (Blueprint $table) {
                $table->id();
                $table->string('order_number')->unique();
                $table->string('customer_name');
                $table->string('customer_phone');
                $table->string('customer_email')->nullable();
                $table->text('customer_address');
                $table->text('delivery_address');
                $table->enum('order_status', ['ORDER_GENERATED', 'ORDER_CREATED', 'PICK_LIST', 'PACKED', 'DISPATCHED', 'DELIVERY', 'MANIFEST', 'DELIVERED', 'RETURNED'])->default('ORDER_GENERATED');
                $table->foreignId('courier_company_id')->nullable()->constrained('courier_companies')->onDelete('set null');
                $table->foreignId('seller_id')->nullable()->constrained('sellers')->onDelete('set null');
                $table->decimal('total_amount', 10, 2)->default(0);
                $table->enum('payment_status', ['pending', 'paid', 'refunded'])->default('pending');
                $table->integer('workspace')->nullable();
                $table->integer('created_by')->default(0);
                $table->timestamps();
            });
        }
    }

    public function down(): void
    {
        Schema::dropIfExists('shipment_orders');
    }
};
```

