<?php

namespace App\Http\Controllers;

use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use App\Models\ShipmentBox;

class BoxController extends Controller
{
    public function index()
    {
        if(\Auth::user()->isAbleTo('box manage'))
        {
            $boxes = ShipmentBox::where('created_by', creatorId())
                ->where('workspace', getActiveWorkSpace())
                ->get();
            
            return view('shipment.boxes.index', compact('boxes'));
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function create()
    {
        if(\Auth::user()->isAbleTo('box create'))
        {
            return view('shipment.boxes.create');
        }
        else
        {
            return response()->json(['error' => __('Permission denied.')], 401);
        }
    }

    public function store(Request $request)
    {
        if(\Auth::user()->isAbleTo('box create'))
        {
            $validator = \Validator::make(
                $request->all(), [
                    'box_number' => 'required|unique:shipment_boxes,box_number',
                    'box_type' => 'required',
                ]
            );
            
            if($validator->fails())
            {
                $messages = $validator->getMessageBag();
                return redirect()->back()->with('error', $messages->first());
            }

            $box = new ShipmentBox();
            $box->box_number = $request->box_number;
            $box->box_type = $request->box_type;
            $box->length = $request->length;
            $box->width = $request->width;
            $box->height = $request->height;
            $box->weight_capacity = $request->weight_capacity;
            $box->price = $request->price ?? 0;
            $box->status = $request->status ?? 'available';
            $box->notes = $request->notes;
            $box->workspace = getActiveWorkSpace();
            $box->created_by = creatorId();
            $box->save();

            return redirect()->route('boxes.index')->with('success', __('Box created successfully.'));
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function edit(ShipmentBox $box)
    {
        if(\Auth::user()->isAbleTo('box edit'))
        {
            if($box->created_by == creatorId() && $box->workspace == getActiveWorkSpace())
            {
                return view('shipment.boxes.edit', compact('box'));
            }
            else
            {
                return response()->json(['error' => __('Permission denied.')], 401);
            }
        }
        else
        {
            return response()->json(['error' => __('Permission denied.')], 401);
        }
    }

    public function update(Request $request, ShipmentBox $box)
    {
        if(\Auth::user()->isAbleTo('box edit'))
        {
            if($box->created_by == creatorId() && $box->workspace == getActiveWorkSpace())
            {
                $validator = \Validator::make(
                    $request->all(), [
                        'box_number' => 'required|unique:shipment_boxes,box_number,' . $box->id,
                        'box_type' => 'required',
                    ]
                );
                
                if($validator->fails())
                {
                    $messages = $validator->getMessageBag();
                    return redirect()->back()->with('error', $messages->first());
                }

                $box->box_number = $request->box_number;
                $box->box_type = $request->box_type;
                $box->length = $request->length;
                $box->width = $request->width;
                $box->height = $request->height;
                $box->weight_capacity = $request->weight_capacity;
                $box->price = $request->price;
                $box->status = $request->status;
                $box->notes = $request->notes;
                $box->save();

                return redirect()->route('boxes.index')->with('success', __('Box updated successfully.'));
            }
            else
            {
                return response()->json(['error' => __('Permission denied.')], 401);
            }
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function destroy(ShipmentBox $box)
    {
        if(\Auth::user()->isAbleTo('box delete'))
        {
            if($box->created_by == creatorId() && $box->workspace == getActiveWorkSpace())
            {
                $box->delete();
                return redirect()->route('boxes.index')->with('success', __('Box deleted successfully.'));
            }
            else
            {
                return response()->json(['error' => __('Permission denied.')], 401);
            }
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }
}

