<?php

namespace App\Http\Controllers;

use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use App\Models\CourierCompany;

class CourierCompanyController extends Controller
{
    /**
     * Display a listing of the resource.
     * @return Renderable
     */
    public function index()
    {
        if(\Auth::user()->isAbleTo('courier company manage'))
        {
            $courierCompanies = CourierCompany::where('created_by', creatorId())
                ->where('workspace', getActiveWorkSpace())
                ->get();
            
            return view('shipment.courier-companies.index', compact('courierCompanies'));
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    /**
     * Show the form for creating a new resource.
     * @return Renderable
     */
    public function create()
    {
        if(\Auth::user()->isAbleTo('courier company create'))
        {
            return view('shipment.courier-companies.create');
        }
        else
        {
            return response()->json(['error' => __('Permission denied.')], 401);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        if(\Auth::user()->isAbleTo('courier company create'))
        {
            $validator = \Validator::make(
                $request->all(), [
                    'name' => 'required',
                ]
            );
            
            if($validator->fails())
            {
                $messages = $validator->getMessageBag();
                return redirect()->back()->with('error', $messages->first());
            }

            $courierCompany = new CourierCompany();
            $courierCompany->name = $request->name;
            $courierCompany->contact_person = $request->contact_person;
            $courierCompany->phone = $request->phone;
            $courierCompany->email = $request->email;
            $courierCompany->address = $request->address;
            $courierCompany->status = $request->status ?? 'active';
            $courierCompany->workspace = getActiveWorkSpace();
            $courierCompany->created_by = creatorId();
            $courierCompany->save();

            return redirect()->route('courier-companies.index')->with('success', __('Courier company created successfully.'));
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(CourierCompany $courierCompany)
    {
        if(\Auth::user()->isAbleTo('courier company edit'))
        {
            if($courierCompany->created_by == creatorId() && $courierCompany->workspace == getActiveWorkSpace())
            {
                return view('shipment.courier-companies.edit', compact('courierCompany'));
            }
            else
            {
                return response()->json(['error' => __('Permission denied.')], 401);
            }
        }
        else
        {
            return response()->json(['error' => __('Permission denied.')], 401);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, CourierCompany $courierCompany)
    {
        if(\Auth::user()->isAbleTo('courier company edit'))
        {
            if($courierCompany->created_by == creatorId() && $courierCompany->workspace == getActiveWorkSpace())
            {
                $validator = \Validator::make(
                    $request->all(), [
                        'name' => 'required',
                    ]
                );
                
                if($validator->fails())
                {
                    $messages = $validator->getMessageBag();
                    return redirect()->back()->with('error', $messages->first());
                }

                $courierCompany->name = $request->name;
                $courierCompany->contact_person = $request->contact_person;
                $courierCompany->phone = $request->phone;
                $courierCompany->email = $request->email;
                $courierCompany->address = $request->address;
                $courierCompany->status = $request->status;
                $courierCompany->save();

                return redirect()->route('courier-companies.index')->with('success', __('Courier company updated successfully.'));
            }
            else
            {
                return response()->json(['error' => __('Permission denied.')], 401);
            }
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(CourierCompany $courierCompany)
    {
        if(\Auth::user()->isAbleTo('courier company delete'))
        {
            if($courierCompany->created_by == creatorId() && $courierCompany->workspace == getActiveWorkSpace())
            {
                $courierCompany->delete();
                return redirect()->route('courier-companies.index')->with('success', __('Courier company deleted successfully.'));
            }
            else
            {
                return response()->json(['error' => __('Permission denied.')], 401);
            }
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }
}

