<?php

namespace App\Http\Controllers;

use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use App\Models\Seller;

class SellerController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        if(\Auth::user()->isAbleTo('seller manage'))
        {
            $sellers = Seller::where('created_by', creatorId())
                ->where('workspace', getActiveWorkSpace())
                ->get();
            
            return view('shipment.sellers.index', compact('sellers'));
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        if(\Auth::user()->isAbleTo('seller create'))
        {
            return view('shipment.sellers.create');
        }
        else
        {
            return response()->json(['error' => __('Permission denied.')], 401);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        if(\Auth::user()->isAbleTo('seller create'))
        {
            $validator = \Validator::make(
                $request->all(), [
                    'name' => 'required',
                ]
            );
            
            if($validator->fails())
            {
                $messages = $validator->getMessageBag();
                return redirect()->back()->with('error', $messages->first());
            }

            $seller = new Seller();
            $seller->name = $request->name;
            $seller->email = $request->email;
            $seller->phone = $request->phone;
            $seller->address = $request->address;
            $seller->company_name = $request->company_name;
            $seller->status = $request->status ?? 'active';
            $seller->workspace = getActiveWorkSpace();
            $seller->created_by = creatorId();
            $seller->save();

            return redirect()->route('sellers.index')->with('success', __('Seller created successfully.'));
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Seller $seller)
    {
        if(\Auth::user()->isAbleTo('seller edit'))
        {
            if($seller->created_by == creatorId() && $seller->workspace == getActiveWorkSpace())
            {
                return view('shipment.sellers.edit', compact('seller'));
            }
            else
            {
                return response()->json(['error' => __('Permission denied.')], 401);
            }
        }
        else
        {
            return response()->json(['error' => __('Permission denied.')], 401);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Seller $seller)
    {
        if(\Auth::user()->isAbleTo('seller edit'))
        {
            if($seller->created_by == creatorId() && $seller->workspace == getActiveWorkSpace())
            {
                $validator = \Validator::make(
                    $request->all(), [
                        'name' => 'required',
                    ]
                );
                
                if($validator->fails())
                {
                    $messages = $validator->getMessageBag();
                    return redirect()->back()->with('error', $messages->first());
                }

                $seller->name = $request->name;
                $seller->email = $request->email;
                $seller->phone = $request->phone;
                $seller->address = $request->address;
                $seller->company_name = $request->company_name;
                $seller->status = $request->status;
                $seller->save();

                return redirect()->route('sellers.index')->with('success', __('Seller updated successfully.'));
            }
            else
            {
                return response()->json(['error' => __('Permission denied.')], 401);
            }
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Seller $seller)
    {
        if(\Auth::user()->isAbleTo('seller delete'))
        {
            if($seller->created_by == creatorId() && $seller->workspace == getActiveWorkSpace())
            {
                $seller->delete();
                return redirect()->route('sellers.index')->with('success', __('Seller deleted successfully.'));
            }
            else
            {
                return response()->json(['error' => __('Permission denied.')], 401);
            }
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }
}

