<?php

namespace App\Http\Controllers;

use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use App\Models\ShipmentInventory;
use App\Models\Warehouse;
use App\Models\StockLocation;
use App\Models\ShipmentItemMaster;

class ShipmentInventoryController extends Controller
{
    public function index()
    {
        if(\Auth::user()->isAbleTo('inventory manage'))
        {
            $inventory = ShipmentInventory::where('workspace', getActiveWorkSpace())
                ->with(['warehouse', 'stockLocation'])
                ->get();
            
            return view('shipment.inventory.index', compact('inventory'));
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function create()
    {
        if(\Auth::user()->isAbleTo('inventory create'))
        {
            $warehouses = Warehouse::where('created_by', creatorId())
                ->where('workspace', getActiveWorkSpace())
                ->get();
            
            $stockLocations = StockLocation::where('created_by', creatorId())
                ->where('workspace', getActiveWorkSpace())
                ->where('status', 'active')
                ->get();
            
            $items = ShipmentItemMaster::where('created_by', creatorId())
                ->where('workspace', getActiveWorkSpace())
                ->where('status', 'active')
                ->get();
            
            return view('shipment.inventory.create', compact('warehouses', 'stockLocations', 'items'));
        }
        else
        {
            return response()->json(['error' => __('Permission denied.')], 401);
        }
    }

    public function store(Request $request)
    {
        if(\Auth::user()->isAbleTo('inventory create'))
        {
            $validator = \Validator::make(
                $request->all(), [
                    'item_sku' => 'required',
                    'warehouse_id' => 'required',
                    'quantity' => 'required|integer|min:0',
                ]
            );
            
            if($validator->fails())
            {
                $messages = $validator->getMessageBag();
                return redirect()->back()->with('error', $messages->first());
            }

            $item = ShipmentItemMaster::where('sku', $request->item_sku)->first();
            
            $inventory = ShipmentInventory::updateOrCreate(
                [
                    'item_sku' => $request->item_sku,
                    'warehouse_id' => $request->warehouse_id,
                    'stock_location_id' => $request->stock_location_id,
                    'workspace' => getActiveWorkSpace(),
                ],
                [
                    'item_name' => $item->item_name ?? $request->item_sku,
                    'quantity' => $request->quantity,
                    'reserved_quantity' => $request->reserved_quantity ?? 0,
                    'min_stock_level' => $request->min_stock_level ?? 0,
                    'max_stock_level' => $request->max_stock_level,
                    'notes' => $request->notes,
                    'created_by' => creatorId(),
                ]
            );

            return redirect()->route('shipment-inventory.index')->with('success', __('Inventory item created successfully.'));
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function edit(ShipmentInventory $shipmentInventory)
    {
        if(\Auth::user()->isAbleTo('inventory edit'))
        {
            if($shipmentInventory->workspace == getActiveWorkSpace())
            {
                $warehouses = Warehouse::where('created_by', creatorId())
                    ->where('workspace', getActiveWorkSpace())
                    ->get();
                
                $stockLocations = StockLocation::where('created_by', creatorId())
                    ->where('workspace', getActiveWorkSpace())
                    ->where('status', 'active')
                    ->get();
                
                return view('shipment.inventory.edit', compact('shipmentInventory', 'warehouses', 'stockLocations'));
            }
            else
            {
                return response()->json(['error' => __('Permission denied.')], 401);
            }
        }
        else
        {
            return response()->json(['error' => __('Permission denied.')], 401);
        }
    }

    public function update(Request $request, ShipmentInventory $shipmentInventory)
    {
        if(\Auth::user()->isAbleTo('inventory edit'))
        {
            if($shipmentInventory->workspace == getActiveWorkSpace())
            {
                $validator = \Validator::make(
                    $request->all(), [
                        'quantity' => 'required|integer|min:0',
                    ]
                );
                
                if($validator->fails())
                {
                    $messages = $validator->getMessageBag();
                    return redirect()->back()->with('error', $messages->first());
                }

                $shipmentInventory->warehouse_id = $request->warehouse_id;
                $shipmentInventory->stock_location_id = $request->stock_location_id;
                $shipmentInventory->quantity = $request->quantity;
                $shipmentInventory->reserved_quantity = $request->reserved_quantity ?? 0;
                $shipmentInventory->min_stock_level = $request->min_stock_level ?? 0;
                $shipmentInventory->max_stock_level = $request->max_stock_level;
                $shipmentInventory->notes = $request->notes;
                $shipmentInventory->save();

                return redirect()->route('shipment-inventory.index')->with('success', __('Inventory item updated successfully.'));
            }
            else
            {
                return response()->json(['error' => __('Permission denied.')], 401);
            }
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function destroy(ShipmentInventory $shipmentInventory)
    {
        if(\Auth::user()->isAbleTo('inventory delete'))
        {
            if($shipmentInventory->workspace == getActiveWorkSpace())
            {
                $shipmentInventory->delete();
                return redirect()->route('shipment-inventory.index')->with('success', __('Inventory item deleted successfully.'));
            }
            else
            {
                return response()->json(['error' => __('Permission denied.')], 401);
            }
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }
}

