<?php

namespace App\Http\Controllers;

use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use App\Models\ShipmentItemMaster;

class ShipmentItemController extends Controller
{
    public function index()
    {
        if(\Auth::user()->isAbleTo('shipment item manage'))
        {
            $items = ShipmentItemMaster::where('created_by', creatorId())
                ->where('workspace', getActiveWorkSpace())
                ->get();
            
            return view('shipment.items.index', compact('items'));
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function create()
    {
        if(\Auth::user()->isAbleTo('shipment item create'))
        {
            return view('shipment.items.create');
        }
        else
        {
            return response()->json(['error' => __('Permission denied.')], 401);
        }
    }

    public function store(Request $request)
    {
        if(\Auth::user()->isAbleTo('shipment item create'))
        {
            $validator = \Validator::make(
                $request->all(), [
                    'item_name' => 'required',
                    'sku' => 'required|unique:shipment_items_master,sku',
                ]
            );
            
            if($validator->fails())
            {
                $messages = $validator->getMessageBag();
                return redirect()->back()->with('error', $messages->first());
            }

            $item = new ShipmentItemMaster();
            $item->item_name = $request->item_name;
            $item->sku = $request->sku;
            $item->description = $request->description;
            $item->weight = $request->weight;
            $item->price = $request->price ?? 0;
            $item->barcode = $request->barcode;
            $item->category = $request->category;
            $item->status = $request->status ?? 'active';
            $item->workspace = getActiveWorkSpace();
            $item->created_by = creatorId();
            $item->save();

            return redirect()->route('shipment-items.index')->with('success', __('Item created successfully.'));
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function edit(ShipmentItemMaster $shipmentItem)
    {
        if(\Auth::user()->isAbleTo('shipment item edit'))
        {
            if($shipmentItem->created_by == creatorId() && $shipmentItem->workspace == getActiveWorkSpace())
            {
                return view('shipment.items.edit', compact('shipmentItem'));
            }
            else
            {
                return response()->json(['error' => __('Permission denied.')], 401);
            }
        }
        else
        {
            return response()->json(['error' => __('Permission denied.')], 401);
        }
    }

    public function update(Request $request, ShipmentItemMaster $shipmentItem)
    {
        if(\Auth::user()->isAbleTo('shipment item edit'))
        {
            if($shipmentItem->created_by == creatorId() && $shipmentItem->workspace == getActiveWorkSpace())
            {
                $validator = \Validator::make(
                    $request->all(), [
                        'item_name' => 'required',
                        'sku' => 'required|unique:shipment_items_master,sku,' . $shipmentItem->id,
                    ]
                );
                
                if($validator->fails())
                {
                    $messages = $validator->getMessageBag();
                    return redirect()->back()->with('error', $messages->first());
                }

                $shipmentItem->item_name = $request->item_name;
                $shipmentItem->sku = $request->sku;
                $shipmentItem->description = $request->description;
                $shipmentItem->weight = $request->weight;
                $shipmentItem->price = $request->price;
                $shipmentItem->barcode = $request->barcode;
                $shipmentItem->category = $request->category;
                $shipmentItem->status = $request->status;
                $shipmentItem->save();

                return redirect()->route('shipment-items.index')->with('success', __('Item updated successfully.'));
            }
            else
            {
                return response()->json(['error' => __('Permission denied.')], 401);
            }
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function destroy(ShipmentItemMaster $shipmentItem)
    {
        if(\Auth::user()->isAbleTo('shipment item delete'))
        {
            if($shipmentItem->created_by == creatorId() && $shipmentItem->workspace == getActiveWorkSpace())
            {
                $shipmentItem->delete();
                return redirect()->route('shipment-items.index')->with('success', __('Item deleted successfully.'));
            }
            else
            {
                return response()->json(['error' => __('Permission denied.')], 401);
            }
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }
}

