<?php

namespace App\Http\Controllers;

use App\DataTables\ShipmentOrderDataTable;
use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Auth;
use App\Models\ShipmentOrder;
use App\Models\ShipmentItem;
use App\Models\CourierCompany;
use App\Models\Seller;
use App\Models\Warehouse;

class ShipmentOrderController extends Controller
{
    /**
     * Display a listing of the resource.
     * @return Renderable
     */
    public function index(ShipmentOrderDataTable $dataTable)
    {
        if(\Auth::user()->isAbleTo('shipment order manage'))
        {
            return $dataTable->render('shipment.orders.index');
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    /**
     * Show the form for creating a new resource.
     * @return Renderable
     */
    public function create()
    {
        if(\Auth::user()->isAbleTo('shipment order create'))
        {
            $courierCompanies = CourierCompany::where('created_by', creatorId())
                ->where('workspace', getActiveWorkSpace())
                ->where('status', 'active')
                ->get();
            
            $sellers = Seller::where('created_by', creatorId())
                ->where('workspace', getActiveWorkSpace())
                ->where('status', 'active')
                ->get();
            
            $warehouses = Warehouse::where('created_by', creatorId())
                ->where('workspace', getActiveWorkSpace())
                ->get();

            return view('shipment.orders.create', compact('courierCompanies', 'sellers', 'warehouses'));
        }
        else
        {
            return response()->json(['error' => __('Permission denied.')], 401);
        }
    }

    /**
     * Store a newly created resource in storage.
     * @param Request $request
     * @return Renderable
     */
    public function store(Request $request)
    {
        if(\Auth::user()->isAbleTo('shipment order create'))
        {
            $validator = \Validator::make(
                $request->all(), [
                    'customer_name' => 'required',
                    'customer_phone' => 'required',
                    'customer_address' => 'required',
                    'delivery_address' => 'required',
                ]
            );
            
            if($validator->fails())
            {
                $messages = $validator->getMessageBag();
                return redirect()->back()->with('error', $messages->first());
            }

            $order = new ShipmentOrder();
            $order->order_number = ShipmentOrder::generateOrderNumber();
            $order->customer_name = $request->customer_name;
            $order->customer_phone = $request->customer_phone;
            $order->customer_email = $request->customer_email;
            $order->customer_address = $request->customer_address;
            $order->delivery_address = $request->delivery_address;
            $order->order_status = $request->order_status ?? 'ORDER_GENERATED';
            $order->courier_company_id = $request->courier_company_id;
            $order->seller_id = $request->seller_id;
            $order->warehouse_id = $request->warehouse_id;
            $order->total_amount = $request->total_amount ?? 0;
            $order->payment_status = $request->payment_status ?? 'pending';
            $order->order_date = $request->order_date ?? now();
            $order->delivery_date = $request->delivery_date;
            $order->notes = $request->notes;
            $order->workspace = getActiveWorkSpace();
            $order->created_by = creatorId();
            $order->save();

            // Save items
            if($request->has('items'))
            {
                foreach($request->items as $itemData)
                {
                    if(!empty($itemData['item_name']))
                    {
                        $item = new ShipmentItem();
                        $item->shipment_order_id = $order->id;
                        $item->item_name = $itemData['item_name'];
                        $item->sku = $itemData['sku'] ?? null;
                        $item->quantity = $itemData['quantity'] ?? 1;
                        $item->weight = $itemData['weight'] ?? null;
                        $item->price = $itemData['price'] ?? 0;
                        $item->total_price = ($itemData['quantity'] ?? 1) * ($itemData['price'] ?? 0);
                        $item->description = $itemData['description'] ?? null;
                        $item->workspace = getActiveWorkSpace();
                        $item->created_by = creatorId();
                        $item->save();
                    }
                }
            }

            return redirect()->route('shipment-orders.index')->with('success', __('Shipment order created successfully.'));
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    /**
     * Show the specified resource.
     * @param int $id
     * @return Renderable
     */
    public function show(ShipmentOrder $shipmentOrder)
    {
        if(\Auth::user()->isAbleTo('shipment order show'))
        {
            if($shipmentOrder->created_by == creatorId() && $shipmentOrder->workspace == getActiveWorkSpace())
            {
                $shipmentOrder->load(['items', 'courierCompany', 'seller', 'warehouse']);
                return view('shipment.orders.show', compact('shipmentOrder'));
            }
            else
            {
                return response()->json(['error' => __('Permission denied.')], 401);
            }
        }
        else
        {
            return response()->json(['error' => __('Permission denied.')], 401);
        }
    }

    /**
     * Show the form for editing the specified resource.
     * @param int $id
     * @return Renderable
     */
    public function edit(ShipmentOrder $shipmentOrder)
    {
        if(\Auth::user()->isAbleTo('shipment order edit'))
        {
            if($shipmentOrder->created_by == creatorId() && $shipmentOrder->workspace == getActiveWorkSpace())
            {
                $courierCompanies = CourierCompany::where('created_by', creatorId())
                    ->where('workspace', getActiveWorkSpace())
                    ->where('status', 'active')
                    ->get();
                
                $sellers = Seller::where('created_by', creatorId())
                    ->where('workspace', getActiveWorkSpace())
                    ->where('status', 'active')
                    ->get();
                
                $warehouses = Warehouse::where('created_by', creatorId())
                    ->where('workspace', getActiveWorkSpace())
                    ->get();

                $shipmentOrder->load('items');

                return view('shipment.orders.edit', compact('shipmentOrder', 'courierCompanies', 'sellers', 'warehouses'));
            }
            else
            {
                return response()->json(['error' => __('Permission denied.')], 401);
            }
        }
        else
        {
            return response()->json(['error' => __('Permission denied.')], 401);
        }
    }

    /**
     * Update the specified resource in storage.
     * @param Request $request
     * @param int $id
     * @return Renderable
     */
    public function update(Request $request, ShipmentOrder $shipmentOrder)
    {
        if(\Auth::user()->isAbleTo('shipment order edit'))
        {
            if($shipmentOrder->created_by == creatorId() && $shipmentOrder->workspace == getActiveWorkSpace())
            {
                $validator = \Validator::make(
                    $request->all(), [
                        'customer_name' => 'required',
                        'customer_phone' => 'required',
                        'customer_address' => 'required',
                        'delivery_address' => 'required',
                    ]
                );
                
                if($validator->fails())
                {
                    $messages = $validator->getMessageBag();
                    return redirect()->back()->with('error', $messages->first());
                }

                $shipmentOrder->customer_name = $request->customer_name;
                $shipmentOrder->customer_phone = $request->customer_phone;
                $shipmentOrder->customer_email = $request->customer_email;
                $shipmentOrder->customer_address = $request->customer_address;
                $shipmentOrder->delivery_address = $request->delivery_address;
                $shipmentOrder->order_status = $request->order_status;
                $shipmentOrder->courier_company_id = $request->courier_company_id;
                $shipmentOrder->seller_id = $request->seller_id;
                $shipmentOrder->warehouse_id = $request->warehouse_id;
                $shipmentOrder->total_amount = $request->total_amount ?? 0;
                $shipmentOrder->payment_status = $request->payment_status;
                $shipmentOrder->order_date = $request->order_date;
                $shipmentOrder->delivery_date = $request->delivery_date;
                $shipmentOrder->notes = $request->notes;
                $shipmentOrder->save();

                // Delete old items
                ShipmentItem::where('shipment_order_id', $shipmentOrder->id)->delete();

                // Save new items
                if($request->has('items'))
                {
                    foreach($request->items as $itemData)
                    {
                        if(!empty($itemData['item_name']))
                        {
                            $item = new ShipmentItem();
                            $item->shipment_order_id = $shipmentOrder->id;
                            $item->item_name = $itemData['item_name'];
                            $item->sku = $itemData['sku'] ?? null;
                            $item->quantity = $itemData['quantity'] ?? 1;
                            $item->weight = $itemData['weight'] ?? null;
                            $item->price = $itemData['price'] ?? 0;
                            $item->total_price = ($itemData['quantity'] ?? 1) * ($itemData['price'] ?? 0);
                            $item->description = $itemData['description'] ?? null;
                            $item->workspace = getActiveWorkSpace();
                            $item->created_by = creatorId();
                            $item->save();
                        }
                    }
                }

                return redirect()->route('shipment-orders.index')->with('success', __('Shipment order updated successfully.'));
            }
            else
            {
                return response()->json(['error' => __('Permission denied.')], 401);
            }
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    /**
     * Remove the specified resource from storage.
     * @param int $id
     * @return Renderable
     */
    public function destroy(ShipmentOrder $shipmentOrder)
    {
        if(\Auth::user()->isAbleTo('shipment order delete'))
        {
            if($shipmentOrder->created_by == creatorId() && $shipmentOrder->workspace == getActiveWorkSpace())
            {
                // Delete items first
                ShipmentItem::where('shipment_order_id', $shipmentOrder->id)->delete();
                
                $shipmentOrder->delete();
                
                return redirect()->route('shipment-orders.index')->with('success', __('Shipment order deleted successfully.'));
            }
            else
            {
                return response()->json(['error' => __('Permission denied.')], 401);
            }
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }
}

