<?php

namespace App\Http\Controllers;

use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use App\Models\StockLocation;
use App\Models\Warehouse;

class StockLocationController extends Controller
{
    public function index()
    {
        if(\Auth::user()->isAbleTo('stock location manage'))
        {
            $stockLocations = StockLocation::where('created_by', creatorId())
                ->where('workspace', getActiveWorkSpace())
                ->with('warehouse')
                ->get();
            
            return view('shipment.stock-locations.index', compact('stockLocations'));
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function create()
    {
        if(\Auth::user()->isAbleTo('stock location create'))
        {
            $warehouses = Warehouse::where('created_by', creatorId())
                ->where('workspace', getActiveWorkSpace())
                ->get();
            
            return view('shipment.stock-locations.create', compact('warehouses'));
        }
        else
        {
            return response()->json(['error' => __('Permission denied.')], 401);
        }
    }

    public function store(Request $request)
    {
        if(\Auth::user()->isAbleTo('stock location create'))
        {
            $validator = \Validator::make(
                $request->all(), [
                    'location_code' => 'required|unique:stock_locations,location_code',
                    'warehouse_id' => 'required',
                ]
            );
            
            if($validator->fails())
            {
                $messages = $validator->getMessageBag();
                return redirect()->back()->with('error', $messages->first());
            }

            $stockLocation = new StockLocation();
            $stockLocation->warehouse_id = $request->warehouse_id;
            $stockLocation->location_code = $request->location_code;
            $stockLocation->shelf_code = $request->shelf_code;
            $stockLocation->zone = $request->zone;
            $stockLocation->description = $request->description;
            $stockLocation->status = $request->status ?? 'active';
            $stockLocation->workspace = getActiveWorkSpace();
            $stockLocation->created_by = creatorId();
            $stockLocation->save();

            return redirect()->route('stock-locations.index')->with('success', __('Stock location created successfully.'));
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function edit(StockLocation $stockLocation)
    {
        if(\Auth::user()->isAbleTo('stock location edit'))
        {
            if($stockLocation->created_by == creatorId() && $stockLocation->workspace == getActiveWorkSpace())
            {
                $warehouses = Warehouse::where('created_by', creatorId())
                    ->where('workspace', getActiveWorkSpace())
                    ->get();
                
                return view('shipment.stock-locations.edit', compact('stockLocation', 'warehouses'));
            }
            else
            {
                return response()->json(['error' => __('Permission denied.')], 401);
            }
        }
        else
        {
            return response()->json(['error' => __('Permission denied.')], 401);
        }
    }

    public function update(Request $request, StockLocation $stockLocation)
    {
        if(\Auth::user()->isAbleTo('stock location edit'))
        {
            if($stockLocation->created_by == creatorId() && $stockLocation->workspace == getActiveWorkSpace())
            {
                $validator = \Validator::make(
                    $request->all(), [
                        'location_code' => 'required|unique:stock_locations,location_code,' . $stockLocation->id,
                        'warehouse_id' => 'required',
                    ]
                );
                
                if($validator->fails())
                {
                    $messages = $validator->getMessageBag();
                    return redirect()->back()->with('error', $messages->first());
                }

                $stockLocation->warehouse_id = $request->warehouse_id;
                $stockLocation->location_code = $request->location_code;
                $stockLocation->shelf_code = $request->shelf_code;
                $stockLocation->zone = $request->zone;
                $stockLocation->description = $request->description;
                $stockLocation->status = $request->status;
                $stockLocation->save();

                return redirect()->route('stock-locations.index')->with('success', __('Stock location updated successfully.'));
            }
            else
            {
                return response()->json(['error' => __('Permission denied.')], 401);
            }
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function destroy(StockLocation $stockLocation)
    {
        if(\Auth::user()->isAbleTo('stock location delete'))
        {
            if($stockLocation->created_by == creatorId() && $stockLocation->workspace == getActiveWorkSpace())
            {
                $stockLocation->delete();
                return redirect()->route('stock-locations.index')->with('success', __('Stock location deleted successfully.'));
            }
            else
            {
                return response()->json(['error' => __('Permission denied.')], 401);
            }
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }
}

