<?php

namespace App\Http\Controllers;

use Illuminate\Contracts\Support\Renderable;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use App\Models\ShipmentVehicle;

class VehicleController extends Controller
{
    public function index()
    {
        if(\Auth::user()->isAbleTo('vehicle manage'))
        {
            $vehicles = ShipmentVehicle::where('created_by', creatorId())
                ->where('workspace', getActiveWorkSpace())
                ->get();
            
            return view('shipment.vehicles.index', compact('vehicles'));
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function create()
    {
        if(\Auth::user()->isAbleTo('vehicle create'))
        {
            return view('shipment.vehicles.create');
        }
        else
        {
            return response()->json(['error' => __('Permission denied.')], 401);
        }
    }

    public function store(Request $request)
    {
        if(\Auth::user()->isAbleTo('vehicle create'))
        {
            $validator = \Validator::make(
                $request->all(), [
                    'vehicle_number' => 'required|unique:shipment_vehicles,vehicle_number',
                    'vehicle_type' => 'required',
                    'driver_name' => 'required',
                    'driver_phone' => 'required',
                ]
            );
            
            if($validator->fails())
            {
                $messages = $validator->getMessageBag();
                return redirect()->back()->with('error', $messages->first());
            }

            $vehicle = new ShipmentVehicle();
            $vehicle->vehicle_number = $request->vehicle_number;
            $vehicle->vehicle_type = $request->vehicle_type;
            $vehicle->driver_name = $request->driver_name;
            $vehicle->driver_phone = $request->driver_phone;
            $vehicle->driver_license = $request->driver_license;
            $vehicle->capacity_weight = $request->capacity_weight;
            $vehicle->capacity_volume = $request->capacity_volume;
            $vehicle->status = $request->status ?? 'active';
            $vehicle->notes = $request->notes;
            $vehicle->workspace = getActiveWorkSpace();
            $vehicle->created_by = creatorId();
            $vehicle->save();

            return redirect()->route('vehicles.index')->with('success', __('Vehicle created successfully.'));
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function edit(ShipmentVehicle $vehicle)
    {
        if(\Auth::user()->isAbleTo('vehicle edit'))
        {
            if($vehicle->created_by == creatorId() && $vehicle->workspace == getActiveWorkSpace())
            {
                return view('shipment.vehicles.edit', compact('vehicle'));
            }
            else
            {
                return response()->json(['error' => __('Permission denied.')], 401);
            }
        }
        else
        {
            return response()->json(['error' => __('Permission denied.')], 401);
        }
    }

    public function update(Request $request, ShipmentVehicle $vehicle)
    {
        if(\Auth::user()->isAbleTo('vehicle edit'))
        {
            if($vehicle->created_by == creatorId() && $vehicle->workspace == getActiveWorkSpace())
            {
                $validator = \Validator::make(
                    $request->all(), [
                        'vehicle_number' => 'required|unique:shipment_vehicles,vehicle_number,' . $vehicle->id,
                        'vehicle_type' => 'required',
                        'driver_name' => 'required',
                        'driver_phone' => 'required',
                    ]
                );
                
                if($validator->fails())
                {
                    $messages = $validator->getMessageBag();
                    return redirect()->back()->with('error', $messages->first());
                }

                $vehicle->vehicle_number = $request->vehicle_number;
                $vehicle->vehicle_type = $request->vehicle_type;
                $vehicle->driver_name = $request->driver_name;
                $vehicle->driver_phone = $request->driver_phone;
                $vehicle->driver_license = $request->driver_license;
                $vehicle->capacity_weight = $request->capacity_weight;
                $vehicle->capacity_volume = $request->capacity_volume;
                $vehicle->status = $request->status;
                $vehicle->notes = $request->notes;
                $vehicle->save();

                return redirect()->route('vehicles.index')->with('success', __('Vehicle updated successfully.'));
            }
            else
            {
                return response()->json(['error' => __('Permission denied.')], 401);
            }
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }

    public function destroy(ShipmentVehicle $vehicle)
    {
        if(\Auth::user()->isAbleTo('vehicle delete'))
        {
            if($vehicle->created_by == creatorId() && $vehicle->workspace == getActiveWorkSpace())
            {
                $vehicle->delete();
                return redirect()->route('vehicles.index')->with('success', __('Vehicle deleted successfully.'));
            }
            else
            {
                return response()->json(['error' => __('Permission denied.')], 401);
            }
        }
        else
        {
            return redirect()->back()->with('error', __('Permission denied.'));
        }
    }
}

