<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class ShipmentOrder extends Model
{
    use HasFactory;

    protected $fillable = [
        'order_number',
        'customer_name',
        'customer_phone',
        'customer_email',
        'customer_address',
        'delivery_address',
        'order_status',
        'courier_company_id',
        'seller_id',
        'warehouse_id',
        'total_amount',
        'payment_status',
        'order_date',
        'delivery_date',
        'notes',
        'workspace',
        'created_by',
    ];

    protected $casts = [
        'order_date' => 'date',
        'delivery_date' => 'date',
        'total_amount' => 'decimal:2',
    ];

    /**
     * Get the items for this shipment order
     */
    public function items()
    {
        return $this->hasMany(ShipmentItem::class);
    }

    /**
     * Get the courier company for this order
     */
    public function courierCompany()
    {
        return $this->belongsTo(CourierCompany::class);
    }

    /**
     * Get the seller for this order
     */
    public function seller()
    {
        return $this->belongsTo(Seller::class);
    }

    /**
     * Get the warehouse for this order
     */
    public function warehouse()
    {
        return $this->belongsTo(Warehouse::class);
    }

    /**
     * Generate unique order number
     */
    public static function generateOrderNumber()
    {
        $prefix = 'SH';
        $lastOrder = self::orderBy('id', 'desc')->first();
        $number = $lastOrder ? $lastOrder->id + 1 : 1;
        return $prefix . str_pad($number, 6, '0', STR_PAD_LEFT);
    }

    /**
     * Scope to filter by workspace
     */
    public function scopeWorkspace($query, $workspace)
    {
        return $query->where('workspace', $workspace);
    }

    /**
     * Scope to filter by status
     */
    public function scopeStatus($query, $status)
    {
        return $query->where('order_status', $status);
    }
}

